/*
 * Lattice Effect
 * Copyright (C) 2009 meatfighter.com
 *
 * This file is part of Lattice Effect.
 *
 * Lattice Effect is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Lattice Effect is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package puls;

import java.awt.*;
import javax.swing.*;
import java.awt.image.*;

public class Main2 extends JFrame {

  private static final int WIDTH = 320;
  private static final int HEIGHT = 200;
  private static final int SCALE = 2;
  private static final boolean FISH_EYE_LENS = false;

  private static final int ITERATIONS = 256;
  private static final float GRID_SIZE = 256.0f;
  private static final float Z0 = 320.0f;
  private static final float VELOCITY_X = 0f;
  private static final float VELOCITY_Y = -20f;
  private static final float VELOCITY_Z = 0f;
  private static final float VELOCITY_THETA = 0.002f;
  private static final float VELOCITY_PHI = 0.004f;
  private static final float VELOCITY_BOLT = 0.05f;

  private static final int DISPLAY_WIDTH = SCALE * WIDTH;
  private static final int DISPLAY_HEIGHT = SCALE * HEIGHT;
  private static final int HALF_WIDTH = WIDTH / 2;
  private static final int HALF_HEIGHT = HEIGHT / 2;
  private static final int PIXELS = WIDTH * HEIGHT;
  private static final float INVERSE_GRID_SIZE = 1.0f / GRID_SIZE;
  private static final float HALF_GRID_SIZE = GRID_SIZE / 2f;
  private static final float NORMAL = (float)(1 / Math.sqrt(3));
  private static final float OCTAHEDRON_SIZE = GRID_SIZE / 4f;
  private static final float BAR_SIZE = GRID_SIZE / 8f;
  private static final float BOLT_SIZE = GRID_SIZE / 5f;
  private static final float BOLT_AMPLITUDE
     = (float)(Math.sqrt(3 * GRID_SIZE * GRID_SIZE) / 2)
         - OCTAHEDRON_SIZE - BOLT_SIZE;
  private static final float BOLD_CENTER
     = (float)(Math.sqrt(3 * GRID_SIZE * GRID_SIZE) / 4)
         + BOLT_SIZE;

  public void launch() {

    int[] pixels = new int[PIXELS];
    float[][] rays = new float[PIXELS][3];

    if (FISH_EYE_LENS) {
      float MAX = Math.max(WIDTH, HEIGHT);
      float X_OFFSET = WIDTH < HEIGHT ? (HEIGHT - WIDTH) / 2 : 0;
      float Y_OFFSET = HEIGHT < WIDTH ? (WIDTH - HEIGHT) / 2 : 0;
      for(int y = 0, k = 0; y < HEIGHT; y++) {
        for(int x = 0; x < WIDTH; x++, k++) {
          float theta = (float)(Math.PI * (0.5f + y + Y_OFFSET) / (float)MAX);
          float phi = (float)(Math.PI * (0.5f + x + X_OFFSET) / (float)MAX);
          float rx = (float)(Math.cos(phi) * Math.sin(theta));
          float ry = (float)(Math.sin(phi) * Math.sin(theta));
          float rz = (float)(Math.cos(theta));
          float[] ray = rays[k];
          ray[0] = rx;
          ray[1] = ry;
          ray[2] = rz;
        }
      }
    } else {
      for(int y = 0, k = 0; y < HEIGHT; y++) {
        for(int x = 0; x < WIDTH; x++, k++) {
          float X = x - HALF_WIDTH + 0.5f;
          float Y = -(y - HALF_HEIGHT + 0.5f);
          float[] ray = rays[k];
          float rx = X;
          float ry = Y;
          float rz = -Z0;
          float inverseMag = 1f / (float)Math.sqrt(rx * rx + ry * ry + rz * rz);
          ray[0] = rx * inverseMag;
          ray[1] = ry * inverseMag;
          ray[2] = rz * inverseMag;
        }
      }
    }

    JPanel panel = (JPanel)getContentPane();
    panel.setPreferredSize(new Dimension(DISPLAY_WIDTH, DISPLAY_HEIGHT));
    setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    setResizable(false);
    pack();
    setLocationRelativeTo(null);
    setVisible(true);

    BufferedImage image = new BufferedImage(
        WIDTH, HEIGHT, BufferedImage.TYPE_INT_RGB);

    float ox = HALF_GRID_SIZE;
    float oy = 0;
    float oz = 0;

    float phi = 0;
    float theta = 0;
    float boltAngle = 0;

    int maxStepT = (int)Math.sqrt(3 * GRID_SIZE * GRID_SIZE);

    int frame = 0;
    long startTime = System.currentTimeMillis() - 1;
    while(true) {

      //float bolt

      float cosPhi = (float)Math.cos(phi);
      float sinPhi = (float)Math.sin(phi);
      float cosTheta = (float)Math.cos(theta);
      float sinTheta = (float)Math.sin(theta);

      float ux = cosPhi * sinTheta;
      float uy = sinPhi * sinTheta;
      float uz = cosTheta;

      float vx = cosPhi * cosTheta;
      float vy = sinPhi * cosTheta;
      float vz = -sinTheta;

      float wx = uy * vz - uz * vy;
      float wy = uz * vx - ux * vz;
      float wz = ux * vy - uy * vx;

      ox += VELOCITY_X;
      oy += VELOCITY_Y;
      oz += VELOCITY_Z;
      theta += VELOCITY_THETA;
      phi += VELOCITY_PHI;
      boltAngle += VELOCITY_BOLT;

      float boltOffset = (float)(BOLD_CENTER
          + BOLT_AMPLITUDE * Math.sin(boltAngle));

      if ((++frame & 0x0F) == 0) {
        setTitle((1000 * frame)
            / (System.currentTimeMillis() - startTime) + " fps");
      }

      int GX = (int)Math.floor(ox * INVERSE_GRID_SIZE);
      int GY = (int)Math.floor(oy * INVERSE_GRID_SIZE);
      int GZ = (int)Math.floor(oz * INVERSE_GRID_SIZE);

      for(int k = 0; k < PIXELS; k++) {

        int red = 0;
        int green = 0;
        int blue = 0;

        int gx = GX;
        int gy = GY;
        int gz = GZ;

        float[] ray = rays[k];
        float Rx = ray[0];
        float Ry = ray[1];
        float Rz = ray[2];

        float rx = ux * Rx + vx * Ry + wx * Rz;
        float ry = uy * Rx + vy * Ry + wy * Rz;
        float rz = uz * Rx + vz * Ry + wz * Rz;

        float irx = 1f / rx;
        float iry = 1f / ry;
        float irz = 1f / rz;

        int dgx = 0;
        int dgy = 0;
        int dgz = 0;

        float tx = 0;
        float ty = 0;
        float tz = 0;

        float dtx = Math.abs(GRID_SIZE * irx);
        float dty = Math.abs(GRID_SIZE * iry);
        float dtz = Math.abs(GRID_SIZE * irz);

        if (rx > 0) {
          dgx = 1;
          tx = ((GRID_SIZE * (gx + 1)) - ox) * irx;
        } else {
          dgx = -1;
          tx = ((GRID_SIZE * gx) - ox) * irx;
        }
        if (ry > 0) {
          dgy = 1;
          ty = ((GRID_SIZE * (gy + 1)) - oy) * iry;
        } else {
          dgy = -1;
          ty = (GRID_SIZE * gy - oy) * iry;
        }
        if (rz > 0) {
          dgz = 1;
          tz = ((GRID_SIZE * (gz + 1)) - oz) * irz;
        } else {
          dgz = -1;
          tz = ((GRID_SIZE * gz) - oz) * irz;
        }

        float t = 0;
        float nextT = Math.min(tx, Math.min(ty, tz));

        int steps = 0;
        for(int i = 0; i < ITERATIONS; i++) {

          float cx = ox - (GRID_SIZE * gx + HALF_GRID_SIZE);
          float cy = oy - (GRID_SIZE * gy + HALF_GRID_SIZE);
          float cz = oz - (GRID_SIZE * gz + HALF_GRID_SIZE);
          
          for(int q = 0; q < ITERATIONS && t < nextT + 32; q++) {
            steps++;
            float x = cx + t * rx;
            float y = cy + t * ry;
            float z = cz + t * rz;

            float x0 = x - HALF_GRID_SIZE;
            float y0 = y - HALF_GRID_SIZE;
            float z0 = z - HALF_GRID_SIZE;
            float x1 = x + HALF_GRID_SIZE;
            float y1 = y + HALF_GRID_SIZE;
            float z1 = z + HALF_GRID_SIZE;
            if (x0 < 0) {
              x0 = -x0;
            }
            if (y0 < 0) {
              y0 = -y0;
            }
            if (z0 < 0) {
              z0 = -z0;
            }
            if (x1 < 0) {
              x1 = -x1;
            }
            if (y1 < 0) {
              y1 = -y1;
            }
            if (z1 < 0) {
              z1 = -z1;
            }

            if (x < 0) {
              x = -x;
            }
            if (y < 0) {
              y = -y;
            }
            if (z < 0) {
              z = -z;
            }
            float a = x - z;
            float b = y - x;
            float c = z - y;
            if (a < 0) {
              a = -a;
            }
            if (b < 0) {
              b = -b;
            }
            if (c < 0) {
              c = -c;
            }

            int stepT = 1;
            boolean centerOctahedron = false;
            boolean cornerOctahedron = false;
            boolean rod = false;
            boolean bolt = false;
            for(int j = 0; j < ITERATIONS; j++) {
              if (x + y + z < OCTAHEDRON_SIZE + stepT) {
                centerOctahedron = true;
                break;
              }
              if (x0 + y0 + z0 < OCTAHEDRON_SIZE + stepT
                  || x0 + y0 + z1 < OCTAHEDRON_SIZE + stepT
                  || x0 + y1 + z0 < OCTAHEDRON_SIZE + stepT
                  || x0 + y1 + z1 < OCTAHEDRON_SIZE + stepT
                  || x1 + y0 + z0 < OCTAHEDRON_SIZE + stepT
                  || x1 + y0 + z1 < OCTAHEDRON_SIZE + stepT
                  || x1 + y1 + z0 < OCTAHEDRON_SIZE + stepT
                  || x1 + y1 + z1 < OCTAHEDRON_SIZE + stepT) {
                cornerOctahedron = true;
                break;
              }
              if (a + b + c < OCTAHEDRON_SIZE + stepT
                  && x + y + z > boltOffset + stepT
                  && x + y + z < boltOffset + BOLT_SIZE + stepT) {
                bolt = true;
                break;
              }
              if (a + b + c < BAR_SIZE + stepT) {
                rod = true;
                break;
              }
              if (stepT >= maxStepT) {
                break;
              }
              stepT <<= 1;
            }
            stepT >>= 1;
            if (stepT == 0) {
              if (centerOctahedron) {
                red = 0;
                green = 8192 / steps;
                if (green > 255) {
                  green = 255;
                }
                blue = 0;
              } else if (cornerOctahedron) {
                red = 8192 / steps;
                if (red > 255) {
                  red = 255;
                }
                green = red >> 1;
                blue = 0;
              } else if (rod) {
                red = 4096 / steps;
                if (red > 255) {
                  red = 255;
                }
                green = red;
                blue = red;
              } else if (bolt) {
                red = 4096 / steps;
                if (red > 255) {
                  red = 255;
                }
                red >>= 1;
                green = red;
                blue = red;
              }
              i = ITERATIONS;
              break;
            }
            t += stepT * NORMAL * 0.5f;
          }
          
          if (tx < ty) {
            if (tx < tz) {
              t = tx;
              tx += dtx;
              gx += dgx;
            } else {
              t = tz;
              tz += dtz;
              gz += dgz;
            }
          } else if (ty < tz) {
            t = ty;
            ty += dty;
            gy += dgy;
          } else {
            t = tz;
            tz += dtz;
            gz += dgz;
          }

          if (tx < ty) {
            if (tx < tz) {
              nextT = tx;
            } else {
              nextT = tz;
            }
          } else if (ty < tz) {
            nextT = ty;
          } else {
            nextT = tz;
          }
        }
        pixels[k] = (red << 16) | (green << 8) | blue;
      }

      Graphics panelGraphics = panel.getGraphics();
      if (panelGraphics != null) {
        image.setRGB(0, 0, WIDTH, HEIGHT, pixels, 0, WIDTH);
        panelGraphics.drawImage(
            image, 0, 0, DISPLAY_WIDTH, DISPLAY_HEIGHT, null);
        panelGraphics.dispose();
      }
    }
  }

  public static void main(String[] args) {
    Main2 main = new Main2();
    main.launch();
  }

}
