/*
 * Bug Game
 * Copyright (C) 2009 meatfighter.com
 *
 * This file is part of Bug Game
 *
 * Bug Game is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Bug Game is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package entomos;

import java.awt.*;
import java.util.*;
import java.awt.image.*;
import java.awt.geom.*;

public class GrubHead implements IObject {

  private boolean[][] occupied;
  private int[][] backgroundTiles;
  private ArrayList<IObject>[] enemies;
  private ArrayList<Bullet> bullets;
  private BufferedImage[] grubHeads;
  private int x;
  private int y;
  private double angle;
  private boolean updated;
  private int dying = -1;
  private int hits = 3;
  private int stunned = 0;
  private AffineTransform transform = new AffineTransform();
  private int shootDelay;

  public void init(boolean[][] occupied, int[][] backgroundTiles,
      ArrayList<IObject>[] enemies, ArrayList<Bullet> bullets,
      int x, int y) {
    this.occupied = occupied;
    this.backgroundTiles = backgroundTiles;
    this.enemies = enemies;
    this.bullets = bullets;
    this.grubHeads = Main.grubHeads;
    this.x = x;
    this.y = y;

    shootDelay = Main.random.nextInt(90);

    for(int i = -2; i <= 1; i++) {
      for(int j = -2; j <= 1; j++) {
        occupied[(y >> 5) + i][(x >> 5) + j] = true;
      }
    }

    enemies[y >> 6].add(this);
  }

  public void prepareForUpdate() {
    updated = false;
  }

  public boolean update(int playerX, int playerY, Bullet playerBullet,
      ArrayList<IObject> enemyList, int enemyIndex) {

    if (updated) {
      return false;
    } else {
      updated = true;
    }

    if (dying > 0) {
      dying--;
      if (dying == 0) {
        for(int i = -2; i <= 1; i++) {
          for(int j = -2; j <= 1; j++) {
            occupied[(y >> 5) + i][(x >> 5) + j] = false;
          }
        }

        enemyList.remove(enemyIndex);
      }
      return false;
    }

    if (stunned > 0) {
      stunned--;
    }

    if (playerBullet != null) {
      double bx = playerBullet.x - x;
      double by = playerBullet.y - y;
      if (bx * bx + by * by <= 1024 && stunned == 0) {
        if (--hits == 0) {
          dying = 15;          
        } else {
          stunned = 10;
        }
        return true;
      }
    }

    double dx = playerX - x;
    double dy = playerY - y;
    if (dx * dx + dy * dy < 4096) {
      Main.addHealth(-1);
    }
    angle = Math.atan2(dy, dx);

    if (shootDelay == 0) {
      shootDelay = 90;
      bullets.add(new Bullet(x, y, playerX, playerY, 5));
    } else {
      shootDelay--;
    }

    return false;
  }

  public void render(Graphics2D g, int cameraX, int cameraY,
      int walkSpriteIndex) {
    if (dying > 10) {
      drawSprite(g, angle, x - cameraX, y - cameraY,
          grubHeads[2]);
    } else if (dying > 0) {
      drawSprite(g, angle, x - cameraX, y - cameraY,
          grubHeads[3]);
    } else if (stunned > 0) {
      drawSprite(g, angle, x - cameraX, y - cameraY,
          grubHeads[1]);
    } else {
      drawSprite(g, angle, x - cameraX, y - cameraY,
          grubHeads[0]);
    }
  }

  private void drawSprite(Graphics2D g,
      double angle, double x, double y, BufferedImage image) {

    double halfWidth = 40;
    double halfHeight = 48;

    transform.setToIdentity();
    transform.translate(x - halfWidth, y - halfHeight);
    transform.rotate(angle, halfWidth, halfHeight);

    g.drawImage(image, transform, null);
  }
}
